// CIniFile.h: interface for the CIniFile class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_CINIFILE_H__788F5DDB_47B0_40F0_B67F_A55BD0B622D3__INCLUDED_)
#define AFX_CINIFILE_H__788F5DDB_47B0_40F0_B67F_A55BD0B622D3__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <string>
#include <vector>

using namespace std;

#define MAX_SECTIONNAME		128
#define MAX_KEYNAME			128
#define MAX_VALUE			2048

class CIniFile  
{
private:
	string path;
  struct section {
    vector<string> keys;
    vector<string> values; 
    vector<string> comments;
  };
  vector<string> keys; 
  vector<string> comments;
  string CheckCase( string s) const;

protected:
  vector<section>    sections; 
  bool   caseInsensitive;

public:
  enum errors{ noID = -1};
  CIniFile( string const iniPath = "");
  virtual ~CIniFile()                            {}

  static string GetApplicationPath();

  // Sets whether or not sectionnames and keynames should be case sensitive.
  // The default is case insensitive.
  void CaseSensitive()                           {caseInsensitive = false;}
  void CaseInsensitive()                         {caseInsensitive = true;}

  // Sets path of ini file to read and write from.
  void Path(string const newPath)                {path = newPath;}
  string Path() const                            {return path;}
  void SetPath(string const newPath)             {Path( newPath);}

  // Reads ini file specified using path.
  // Returns true if successful, false otherwise.
  bool ReadFile();
  
  // Writes data stored in class to ini file.
  bool WriteFile(); 
  
  // Deletes all stored ini data.
  void Erase();
  void Clear()                                   {Erase();}
  void Reset()                                   {Erase();}

  // Returns index of specified section, or noID if not found.
  long FindSection( string const sectionname) const;

  // Returns index of specified key, in the specified section, or noID if not found.
  long FindKey( unsigned const sectionID, string const keyname) const;

  // Returns number of sections currently in the ini.
  unsigned NumSections() const                       {return keys.size();}
  unsigned GetNumSections() const                    {return NumSections();}

  // Add a section name.
  unsigned AddSectionName( string const sectionname);

  // Returns section names by index.
  string SectionName( unsigned const sectionID) const;
  string GetSectionName( unsigned const sectionID) const {return SectionName(sectionID);}

  // Returns number of keys stored for specified section.
  unsigned NumKeys( unsigned const sectionID);
  unsigned GetNumKeys( unsigned const sectionID)   {return NumKeys( sectionID);}
  unsigned NumKeys( string const sectionname);
  unsigned GetNumKeys( string const sectionname)   {return NumKeys( sectionname);}

  // Returns key name by index for a given sectionname or sectionID.
  string KeyName( unsigned const sectionID, unsigned const keyID) const;
  string GetKeyName( unsigned const sectionID, unsigned const keyID) const {
    return KeyName( sectionID, keyID);
  }
  string KeyName( string const sectionname, unsigned const keyID) const;
  string GetKeyName( string const sectionname, unsigned const keyID) const {
    return KeyName( sectionname, keyID);
  }

  // Sets value of [sectionname] keyname =.
  bool SetValue( unsigned const sectionID, unsigned const keyID, string const value);
  bool SetValue( string const sectionname, string const keyname, string const value, bool const create = true);

  // Deletes specified key.
  // Returns true if key existed and deleted, false otherwise.
  bool DeleteKey( string const sectionname, string const keyname);
  
  // Deletes specified section and all keys contained within.
  // Returns true if section existed and deleted, false otherwise.
  bool DeleteSection(string sectionname);

  // Header comment functions.
  // Header comments are those comments before the first section.
  //
  // Number of header comments.
  unsigned NumHeaderComments()                  {return comments.size();}
  // Add a header comment.
  void     HeaderComment( string const comment);
  // Return a header comment.
  string   HeaderComment( unsigned const commentID) const;
  // Delete a header comment.
  bool     DeleteHeaderComment( unsigned commentID);
  // Delete all header comments.
  void     DeleteHeaderComments()               {comments.clear();}

  // Section comment functions.
  // Section comments are those comments within a section. Any comments
  // defined within value names will be added to this list. Therefore,
  // these comments will be moved to the top of the section definition when
  // the CIniFile::WriteFile() is called.
  //
  // Number of section comments.
  unsigned NumSectionComments( unsigned const sectionID) const;
  unsigned NumSectionComments( string const sectionname) const;
  // Add a section comment.
  bool     SectionComment( unsigned const sectionID, string const comment);
  bool     SectionComment( string const sectionname, string const comment);
  // Return a section comment.
  string   SectionComment( unsigned const sectionID, unsigned const commentID) const;
  string   SectionComment( string const sectionname, unsigned const commentID) const;
  // Delete a section comment.
  bool     DeleteSectionComment( unsigned const sectionID, unsigned const commentID);
  bool     DeleteSectionComment( string const sectionname, unsigned const commentID);
  // Delete all comments for a section.
  bool     DeleteSectionComments( unsigned const sectionID);
  bool     DeleteSectionComments( string const sectionname);
};

#endif // !defined(AFX_CINIFILE_H__788F5DDB_47B0_40F0_B67F_A55BD0B622D3__INCLUDED_)
