// CIniFile.cpp: implementation of the CIniFile class.
//
//////////////////////////////////////////////////////////////////////

#include <afx.h>
#include <afxwin.h>
#include <fstream>

#include "CIniFile.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define iniEOL endl

string CIniFile::GetApplicationPath()
{
	// Read Application path of Registry
	TCHAR   sFilePath[255]; 

	// Direct way to get the application title.
	CString sFileName = AfxGetAppName();			/* C:\AMVER_SEAS\CPP_SOURCE\PCWatchDOg\ */
	CString sFileNameExt = sFileName +".ini"; 
	LPCTSTR lpDefault = TEXT("Error: GPPS failed");
	GetPrivateProfileString(TEXT(sFileName),		// pointer to section name 
							TEXT("ApplicationPath"),// pointer to key name
							lpDefault,				// points to default string
							sFilePath,				// points to destination buffer
							255,					// size of destination buffer
							TEXT(sFileNameExt));	// points to initialization filename 

	if( strcmp( sFilePath, lpDefault ) == 0 )
	{
		string sEmptyPath = "";
		return sEmptyPath;
	};
	return sFilePath;
};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CIniFile::CIniFile( string const iniPath)
{
  Path( iniPath);
  caseInsensitive = true;
  ReadFile();
}

bool CIniFile::ReadFile()
{
  // Normally you would use ifstream, but the SGI CC compiler has
  // a few bugs with ifstream. So ... fstream used.
  fstream f;
  string   line;
  string   sectionname, valuename, value;
  string::size_type pLeft, pRight;

  f.open( path.c_str(), ios::in);
  if ( f.fail())
    return false;
  
  while( getline( f, line)) {
    // To be compatible with Win32, check for existence of '\r'.
    // Win32 files have the '\r' and Unix files don't at the end of a line.
    // Note that the '\r' will be written to INI files from
    // Unix so that the created INI file can be read under Win32
    // without change.
    if ( line[line.length() - 1] == '\r')
      line = line.substr( 0, line.length() - 1);
    
    if ( line.length()) {
      // Check that the user hasn't openned a binary file by checking the first
      // character of each line!
      if ( !isprint( line[0])) {
	printf( "Failing on char %d\n", line[0]);
	f.close();
	return false;
      }
      if (( pLeft = line.find_first_of(";#[=")) != string::npos) {
	switch ( line[pLeft]) {
	case '[':
	  if ((pRight = line.find_last_of("]")) != string::npos &&
	      pRight > pLeft) {
	    sectionname = line.substr( pLeft + 1, pRight - pLeft - 1);
	    AddSectionName( sectionname);
	  }
	  break;
	  
	case '=':
	  valuename = line.substr( 0, pLeft);
	  value = line.substr( pLeft + 1);
	  SetValue( sectionname, valuename, value);
	  break;
	  
	case ';':
	case '#':
	  if ( !keys.size())
	    HeaderComment( line.substr( pLeft + 1));
	  else
	    SectionComment( sectionname, line.substr( pLeft + 1));
	  break;
	}
      }
    }
  }

  f.close();
  if ( keys.size())
    return true;
  return false;
}

bool CIniFile::WriteFile()
{
  unsigned commentID, sectionID, valueID;
  // Normally you would use ofstream, but the SGI CC compiler has
  // a few bugs with ofstream. So ... fstream used.
  fstream f;

  f.open( path.c_str(), ios::out);
  if ( f.fail())
    return false;

  // Write header comments.
  for ( commentID = 0; commentID < comments.size(); ++commentID)
    f << ';' << comments[commentID] << iniEOL;
  if ( comments.size())
    f << iniEOL;

  // Write sections and values.
  for ( sectionID = 0; sectionID < sections.size(); ++sectionID) {
    f << '[' << keys[sectionID] << ']' << iniEOL;
    // Comments.
    for ( commentID = 0; commentID < sections[sectionID].comments.size(); ++commentID)
      f << ';' << sections[sectionID].comments[commentID] << iniEOL;
    // Values.
    for ( valueID = 0; valueID < sections[sectionID].keys.size(); ++valueID)
      f << sections[sectionID].keys[valueID] << '=' << sections[sectionID].values[valueID] << iniEOL;
    f << iniEOL;
  }
  f.close();
  
  return true;
}

long CIniFile::FindSection( string const sectionname) const
{
  for ( unsigned sectionID = 0; sectionID < keys.size(); ++sectionID)
    if ( CheckCase( keys[sectionID]) == CheckCase( sectionname))
      return long(sectionID);
  return noID;
}

long CIniFile::FindKey( unsigned const sectionID, string const keyname) const
{
  if ( !sections.size() || sectionID >= sections.size())
    return noID;

  for ( unsigned keyID = 0; keyID < sections[sectionID].keys.size(); ++keyID)
    if ( CheckCase( sections[sectionID].keys[keyID]) == CheckCase( keyname))
      return long(keyID);
  return noID;
}

unsigned CIniFile::AddSectionName( string const sectionname)
{
  keys.resize( keys.size() + 1, sectionname);
  sections.resize( sections.size() + 1);
  return keys.size() - 1;
}

string CIniFile::SectionName( unsigned const sectionID) const
{
  if ( sectionID < keys.size())
    return keys[sectionID];
  else
    return "";
}

unsigned CIniFile::NumKeys( unsigned const sectionID)
{
  if ( sectionID < sections.size())
    return sections[sectionID].keys.size();
  return 0;
}

unsigned CIniFile::NumKeys( string const sectionname)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return 0;
  return sections[sectionID].keys.size();
}

string CIniFile::KeyName( unsigned const sectionID, unsigned const keyID) const
{
  if ( sectionID < sections.size() && keyID < sections[sectionID].keys.size())
    return sections[sectionID].keys[keyID];
  return "";
}

string CIniFile::KeyName( string const sectionname, unsigned const keyID) const
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return "";
  return KeyName( sectionID, keyID);
}

bool CIniFile::SetValue( unsigned const sectionID, unsigned const keyID, string const value)
{
  if ( sectionID < sections.size() && keyID < sections[sectionID].keys.size())
    sections[sectionID].values[keyID] = value;

  return false;
}

bool CIniFile::SetValue( string const sectionname, string const keyname, string const value, bool const create)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID) {
    if ( create)
      sectionID = long( AddSectionName( sectionname));
    else
      return false;
  }

  long keyID = FindKey( unsigned(sectionID), keyname);
  if ( keyID == noID) {
    if ( !create)
      return false;
    sections[sectionID].keys.resize( sections[sectionID].keys.size() + 1, keyname);
    sections[sectionID].values.resize( sections[sectionID].values.size() + 1, value);
  } else
    sections[sectionID].values[keyID] = value;

  return true;
}

bool CIniFile::DeleteKey( string const sectionname, string const keyname)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return false;

  long keyID = FindKey( unsigned(sectionID), keyname);
  if ( keyID == noID)
    return false;

  // This looks strange, but is neccessary.
  vector<string>::iterator kpos = sections[sectionID].keys.begin() + keyID;
  vector<string>::iterator vpos = sections[sectionID].values.begin() + keyID;
  sections[sectionID].keys.erase( kpos, kpos + 1);
  sections[sectionID].values.erase( vpos, vpos + 1);

  return true;
}

bool CIniFile::DeleteSection( string const sectionname)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return false;

  // Now hopefully this destroys the vector lists within sections.
  // Looking at <vector> source, this should be the case using the destructor.
  // If not, I may have to do it explicitly. Memory leak check should tell.
  // memleak_test.cpp shows that the following not required.
  //sections[sectionID].keys.clear();
  //sections[sectionID].values.clear();

  vector<string>::iterator npos = keys.begin() + sectionID;
  vector<section>::iterator    kpos = sections.begin() + sectionID;
  keys.erase( npos, npos + 1);
  sections.erase( kpos, kpos + 1);

  return true;
}

void CIniFile::Erase()
{
  // This loop not needed. The vector<> destructor seems to do
  // all the work itself. memleak_test.cpp shows this.
  //for ( unsigned i = 0; i < sections.size(); ++i) {
  //  sections[i].keys.clear();
  //  sections[i].values.clear();
  //}
  keys.clear();
  sections.clear();
  comments.clear();
}

void CIniFile::HeaderComment( string const comment)
{
  comments.resize( comments.size() + 1, comment);
}

string CIniFile::HeaderComment( unsigned const commentID) const
{
  if ( commentID < comments.size())
    return comments[commentID];
  return "";
}

bool CIniFile::DeleteHeaderComment( unsigned commentID)
{
  if ( commentID < comments.size()) {
    vector<string>::iterator cpos = comments.begin() + commentID;
    comments.erase( cpos, cpos + 1);
    return true;
  }
  return false;
}

unsigned CIniFile::NumSectionComments( unsigned const sectionID) const
{
  if ( sectionID < sections.size())
    return sections[sectionID].comments.size();
  return 0;
}

unsigned CIniFile::NumSectionComments( string const sectionname) const
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return 0;
  return sections[sectionID].comments.size();
}

bool CIniFile::SectionComment( unsigned const sectionID, string const comment)
{
  if ( sectionID < sections.size()) {
    sections[sectionID].comments.resize( sections[sectionID].comments.size() + 1, comment);
    return true;
  }
  return false;
}

bool CIniFile::SectionComment( string const sectionname, string const comment)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return false;
  return SectionComment( unsigned(sectionID), comment);
}

string CIniFile::SectionComment( unsigned const sectionID, unsigned const commentID) const
{
  if ( sectionID < sections.size() && commentID < sections[sectionID].comments.size())
    return sections[sectionID].comments[commentID];
  return "";
}

string CIniFile::SectionComment( string const sectionname, unsigned const commentID) const
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return "";
  return SectionComment( unsigned(sectionID), commentID);
}

bool CIniFile::DeleteSectionComment( unsigned const sectionID, unsigned const commentID)
{
  if ( sectionID < sections.size() && commentID < sections[sectionID].comments.size()) {
    vector<string>::iterator cpos = sections[sectionID].comments.begin() + commentID;
    sections[sectionID].comments.erase( cpos, cpos + 1);
    return true;
  }
  return false;
}

bool CIniFile::DeleteSectionComment( string const sectionname, unsigned const commentID)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return false;
  return DeleteSectionComment( unsigned(sectionID), commentID);
}

bool CIniFile::DeleteSectionComments( unsigned const sectionID)
{
  if ( sectionID < sections.size()) {
    sections[sectionID].comments.clear();
    return true;
  }
  return false;
}

bool CIniFile::DeleteSectionComments( string const sectionname)
{
  long sectionID = FindSection( sectionname);
  if ( sectionID == noID)
    return false;
  return DeleteSectionComments( unsigned(sectionID));
}

string CIniFile::CheckCase( string s) const
{
  if ( caseInsensitive)
    for ( string::size_type i = 0; i < s.length(); ++i)
      s[i] = tolower(s[i]);
  return s;
}
