       SUBROUTINE RADARDISCREAD1(LUT,KINDRADAR,IHEADERFLAG,IDATAFLAG,
     +                          IERR)
C
C   This subroutine reads and unpacks the 1988 Radar format data.  
C The data is read from a disc file. The routine
C returns 1 ray of data from one of the radars per call.  Radardiscread
C follows the SIGMET documentation.  It returns the Tape Header information,
C the Data Record Header information, the Ray Header information and the data.
C The Tape Header is a 1024 record, the Data record is variable length with
C a maximum size of 4096 words, the Ray Header which is part of the Data
C record is 22 words long and the data contains a maximum of 512 bins.
C IDRADAR contains the radar being read: 1 for Lower Fuselage and
C 2 for Tail.  LF stands for Lower Fuselage, TA stands for Tail.
C The Tape Header contains information from both radars.
C    The calling program sends:
C    LUT - unit number connected to the file
C    KINDRADAR - kind of radar to decode, 0=both, 1=LF, 2=TA (integer)
C    IHEADERFLAG - flag that determines if the user wants the 1024 words
C                  contained in the Tape Header, 0=no, 1=yes.
C                  A subset is returned in the TAPEHEADER common area.(integer)
C    IDATAFLAG - flag that determines the type of data to be returned
C                to the calling program: 
C              bit 2 on: returns dbz
C              bit 1 on: returns velocity    
C              bit 0 on: returns spectral width
C    Radardiscread returns:
C    IERR - error code
C            0 no error; good read
C           -2 end of file(final record on the file)
C    
C    Radardiscread returns two common areas.
C    TAPEHEADER - common area containing some parameters from the data.
C                 Header and a buffer(IHEADER) that contains the 1024 words.
C                 The information in IHEADER is scaled according to the
C                 SIGMET documentation.  The routine returns buffer IHEADER
C                 if IHEADERFLAG equals 1.  IHEADER is set to 0 otherwise.
C                 TAPEHEADER is composed of the following:
C    nbytesheader - number of bytes in the Tape Header.
C    ntape - tape number.
C    nversion - Tape Header format version number.
C    ihheader,imheader,isheader - Tape Header time in hours, minutes 
C                                 and seconds.
C    ine - navigation system used 0:ONE, 1:INE1, 2:INE2
C    ndrive - drive number 1 or 2.
C    naircraft - aircraft identifier 42, 43 or 0 for ground.
C    flightid - character variable containing the flight identifier (8 char).
C    stmname - project identifier (16 characters).    
C    refslopelf,refslopeta - reflectivity slope for LF and TA.
C    refnoisethreslf,refnoisethresta -refl. noise threshold for LF and TA.
C    sqilf,sqita - SQI threshold for LF and TA.
C    widththreslf,widththresta - power threshold for width; LF and TA.
C    caliblf,calibta - calibration refl. for LF and TA.
C    modedatalf,modedatata - data mode for LF and TA; 1=processed data,
C                                                     2=time series.
C    noisetmlf,noisetmta - time between noise samples in minutes for LF & TA.
C    rpmlf,rpmta - antenna sweep speed for LF and TA.
C    gtlnlf - gate length for the LF radar in km.
C    ivrangelf - flag for LF variable range gates (1=var,0=fix).
C    ivrangeta - flag for TA variable range gates (1=var,0=fix).
C    gtlnta(3) - 3 dimensional array containing Tail gate length values in km;
C                for variable gtlnta(1)=0.075,gtlnta(2)=0.150,gtlnta(3)=.300
C                for fixed gtlnta(1)=value used,gtlnta(2)=0.0,gtlnta(3)=0.0.
C    rdellf,rdelta - range delay in km for LF and TA.
C    rdelcorta,rdelcorlf - range delay correction value for TA and LF,
C                          passed by the calling program.
C    nbinlf,nbinta - number of output bins for LF and TA.
C    nbins - number of output bins for radar read.
C    rmaxlf - maximum range of the LF radar.  Rmaxlf is not provided on the
C             tapes,it is computed as: rmaxlf=float(nbinlf)*gtlnlf + rdellf.
C    rmaxta - maximum range of the TA radar.  Rmaxta is not recorded on the
C             tapes, it is computed as:
C    fixed gates rmaxta=float(nbinta)*gtlnta(1) + rdelta
C    variable gates rmaxta=256.*gtlnta(1)+128.*gtlnta(2)+128.*gtlnta(3)+rdelta
C    radiita - 512 array containing Tail data ranges.
C    radiilf - 512 array containing LF data ranges.
C    prf_flaglf - mode, 0:single prf, 1:dual prf 2/3, 2:dual prf 3/4
C    prf_flagta - mode, 0:single prf, 1:dual prf 2/3, 2:dual prf 3/4
C    prflf, prfta - prf recorded on the tape for LF and Tail Doppler.
C    xniqlf, xniqta - Max. Nyquist velocity for Tail and Lf Dopplers.
C    iheader - 1024 word array containing the Tape Header parameters.
C    
C    RAYHEADER - common area containing the Data Record Header information,
C                the Ray Header information and the data.  The data is 
C                returned in 3 arrays: DBZBUFFER(dbz), VELBUFFER(velocity)
C                and WIDTHBUFFER(spectral width).  The number of gates
C                returned is NBINS with a maximum of 512 bins.  The data
C                has been unpacked, uncompressed and converted to dbz,
C                folded velocity and spectral width.
C                IDRADAR returns the kind of radar being read:1:LF,2:TA.
C                RAYHEADER contains information from one of the radars and
C                is composed of the following:
C    nbytesdata - size of data record in bytes.
C    nlfsweep, ntasweep - LF sweep number, TA sweep number.
C    numrecord - record number in the sweep for that radar.
C    irecflag - record number flag(1=first,0=middle,2=last)
C    idradar - Radar #: 1:LF, 2:TA..
C    nbytesray - number of bytes in ray header.
C    icode - data set code value.  It is used by the compression algorithm.
C            bit 15 is on if the ray contains reflectivity
C            bit 14 is on if the ray contains velocity    
C            bit 13 is on if the ray contains spectral width
C            bits 11&12 hold a code for which DSP the data is from(1:LF,2:TA)
C            bit 10 is on if the ray contains time series data.
C    idsp - code for which DSP the data is from.  This code is 1 for the
C           LF and 2 for the TA radar.  Bits 11 and 12 on icode.
C    mdata - data mode code.  Code is 1 for time series and 2 for processed
C            data.  Bit 10 on icode.
C    iyear,month,iday - year, month and day of data.
C    iraycode - ray code.
C    time - time of data in seconds.  Radardiscread returns a 48 hour clock.
C           The time in the tapes is set to 00:00:00 at midnight.
C    ih - hours(48 hour)
C    im - minutes
C    is - seconds
C    is100 - hundred of seconds
C    rlat - radar latitude (deg).
C    rlon - radar longitude(deg).
C    ra - radar altitude(meters).
C    gse - E/W ground speed (m/s).
C    gsn _ N/S ground speed (m/s).
C    gs - ground speed.  GS is not recorded as such on the tapes.  It
C         is computed: GS=SQRT(GSN*GSN + GSE*GSE) (m/s).
C    vgs - vertical ground speed (m/s).
C    ucom - wind velocity, East(m/s).
C    vcom - wind velocity, North(m/s).
C    wd - wind direction.  WD is not recorded on the tapes.  It is computed
C         by WDIR (function on biglib) (deg).
C    ws - wind speed.  WS is not recorded on the tapes.  It is computed by
C         SPEED (function on biglib) (m/s).
C    vws - vertical wind speed.
C    ircu - rcu status word.
C    elev - elevation angle (deg).
C    azm - azimuth (deg).
C    pitch - pitch angle (deg).
C    roll - roll angle (deg).
C    drift - drift angle (deg).
C    heading - aircraft heading (deg).
C    dbzbuffer - array containing dbz values.
C                The overall range is from -31.5 to +95.5 dbz in half dB
C                steps.  -32.0 indicates no reflectivity data available at 
C                that bin number.
C                It contains a maximum of 512 but only NBINS are returned.
C    velbuffer - array containing folded velocities.
C                Mean velocity is with respect to the unambiguous interval.
C                  1: Maximum velocity towards the radar.
C                128: Zero velocity
C                255: Maimum velocity away from the radar.  
C             -888.8: indicates velocity not available at this range.
C                It contains a maximum of 512 but only NBINS are returned.
C    widthbuffer - array containing spectral width.
C                The overall range is a number between (1/256)*xniqta and
C                (255/256)*xniqta meters per second.  XNIQTA is the nyquist 
C                velocity.  0.0 indicates that width was not available at
C                this range.  
C                It contains a maximum of 512 but only NBINS are returned.
C
      COMMON /TAPEHEADER/ nbytesheader,ntape,nversion,ihheader,imheader,
     +                    isheader,ine,ndrive,naircraft,flightid,stmname
     +                    ,nsamplelf,nsampleta,refslopelf,refslopeta,
     +                    refnoisethreslf,refnoisethresta,sqilf,sqita,
     +                    widththreslf,widththresta,caliblf,calibta,
     +                    modedatalf,modedatata,noisetmlf,noisetmta,
     +                    rpmlf,rpmta,gtlnlf,ivrangelf,ivrangeta,
     +                    gtlnta(3),rdellf,rdelta,rdelcorlf,rdelcorta,
     +                    nbinlf,nbinta,nbins,rmaxlf,rmaxta,
     +                    radiita(512),radiilf(512),prf_flaglf,
     +                    prf_flagta,prf_highlf,prf_highta,xniqlf,
     +                    xniqta,iheader(1024)
c
      common /rayheader / nbytesdata,nlfsweep,ntasweep,numrecord,
     +                    irecflag,idradar,nbytesray,icode,idsp,mdata,
     +                    iyear,month,iday,iraycode,time,ih,im,
     +                    is,is100,rlat,rlon,ra,gse,gsn,gs,vgs,
     +                    ucom,vcom,wd,ws,vws,ircu,
     +                    elev,azm,pitch,roll,drift,heading,
     +                    dbzbuffer(512),velbuffer(512),widthbuffer(512)
      common /tsearch/ kountrec,firstsweep(2),keylf(1024,2),
     +                 keyta(4096,2),
     +                 numsweep(2),itime(2),lastpoint(2),
     +                 iheadertime,ioldsweep(2),kwords,nwords,krays,buf
      COMMON /IERRINONE/ ierrin
      integer*2 buf(4096) , iheader
      integer iz(512),iv(512),iw(512) 
      integer firstsweep
      integer prf_flaglf, prf_flagta
      integer iprf(0:2)
      real lamda_lf, lamda_ta
      character flightid*8,stmname*16 
      character nameradarfile*(*), jfile*60
      logical search, search1
      logical first
      save kwordsave,lamda_lf, lamda_ta, velconv_ta

      data kwords/0/, nwords/-1/, krays/0/, binary/0.005493164/,
     +     ioldsweep/2*0/, iheader/1024*0/, nix/0/, iprf/1,2,3/
      data gtlnlf/-999./, ivrangelf/0/, 
     +     gtlnta/-999.,-999.,-999./
     +     ivrangeta/0/, rdellf/0.0/, rdelta/0.0/, rdelcorlf/0.0/,
     +     rdelcorta/0.0/, nbins/0/, nbinlf/0/, nbinta/0/,
     +     rmaxlf/-999./, rmaxta/-999./
      data dbzbuffer/512*-32.0/, velbuffer/512*-888.88/, 
     +     widthbuffer/512*0.0/
      data kountrec/0/, firstsweep/2*0/, keylf/2048*0/, keyta/8192*0/,
     +     numsweep/2*0/, itime/2*-1/, lastpoint/1024,4096/
      data first/.true./, maxlength/8192/
c 
c read a disc record
      search= .false.
      go to 50
      entry discentry(lut,kindradar,iheaderflag,search1,ierr)!from discsearch
c      write(6,*)'ierrin = ',ierrin
c      write(6,*)'search1 = ',search1
      search=search1
      if(ierrin.eq.20)go to 100
50    if (kwords .lt. nwords) go to 300 
c      write(6,*)'calling diskread1'
100   call diskread1(lut,first,buf,maxlength,length)!c routine to read from disc
c      write(6,*)'called diskread1'
      ierr=length
      if(ierr.lt.0)then
        write(6,'(i4," End of disc file occurred:",3i2.2)')ierr,ih,im,is
        return 
      endif
      if (ierr.eq.2048 .and. buf(1).eq.0) then !header record. 
         write(6,'(" tape header record read at ",3i2.2,/,1x,4a2,
     +         1x,"tape# ",i4)')
     +        buf(9),buf(10),buf(11),(buf(i),i=40,43), buf(3)
         nix=0 
c    convert header time from hours, minutes and seconds into seconds
         iheadertime=buf(9)*3600 + buf(10)*60 + buf(11)
c    # of records between tape headers
         kountrec=0
         do i=1,2
            firstsweep(i)=0
            itime(i)= -1
            numsweep(i)=0
            ioldsweep(i)= 0
         end do
         if (iheaderflag .eq. 1) then
            do i=1, 1024
               iheader(i)=buf(i)
            end do
         endif
c     number of bytes in tape header
         nbytesheader=buf(2)
c     tape number 
         ntape=buf(3)
c     format version number
         nversion=buf(4)
c     tape header time in hours, minutes and seconds
         ihheader=buf(9)
         imheader=buf(10)
         isheader=buf(11)
c     ine 
         ine=buf(37) 
c     drive number
         if (buf(38).eq.8) ndrive=1
         if (buf(38).eq.9) ndrive=2
c     aircraft identifier (42 or 43)
         naircraft=buf(39) 
c      flight id number 
         write(flightid,'(4a2)') (buf(i),i=40,43) 
c      storm name
         write(stmname,'(8a2)') (buf(i),i=81,88)
c      sample size
         nsamplelf=buf(101)
         nsampleta=buf(401)
c      dsp flags, prf mode
         prf_flaglf=ibits(buf(102),8,2)
         prf_flagta=ibits(buf(402),8,2)
c      store calibration information for the lf and ta respectively 
c      reflectivity slope 
         refslopelf=float(buf(103))/4096.   
         refslopeta=float(buf(403))/4096. 
c      reflectivity noise threshold 
         refnoisethreslf=float(buf(104))/16.         
         refnoisethresta=float(buf(404))/16.     
c      sqi threshold
         sqilf=float(buf(106))/256.
         sqita=float(buf(406))/256.
c      power threshold for width
         widththreslf=float(buf(107))/16. 
         widththresta=float(buf(407))/16. 
c      calibration reflectivity 
         caliblf=float(buf(108))/16. 
         calibta=float(buf(408))/16. 
c      data mode 1:processed data, 2:time series
         modedatalf=buf(115)
         modedatata=buf(415)
c      time between noise samples for the lower fuselage and tail
         noisetmlf=buf(159)
         noisetmta=buf(459)
c      wavelength in 1/100 cm
         lamda_lf=float(buf(172))/10000.0
         lamda_ta=float(buf(472))/10000.0
c      prf recorded on tape
         PRF_HIGHLF=FLOAT(BUF(174))
         PRF_HIGHTA=FLOAT(BUF(474))
C      calculated from the prf mode, virtual prf
         PRF_VIRTUALLF=PRF_HIGHLF * IPRF(PRF_FLAGLF)
         PRF_VIRTUALTA=PRF_HIGHTA * IPRF(PRF_FLAGTA)
C      maximum nyquist velocity in m/s
         XNIQLF=(PRF_VIRTUALLF * LAMDA_LF)/4.0
         XNIQTA=(PRF_VIRTUALTA * LAMDA_TA)/4.0
C      velocity conversion factor for the tail/Doppler data
         VELCONV_TA=XNIQTA/128.0
C      LF antenna sweep speed in 1/10 of RPM, signed
         RPMLF=ABS(FLOAT(BUF(192))*0.1) 
C      TA antenna sweep speed in 1/10 of RPM, signed
         RPMTA=ABS(FLOAT(BUF(492))*0.1) 
C      range delay for the LF 
         RDELLF=FLOAT(BUF(141)) + FLOAT(BUF(146))/10000.0 + RDELCORLF
C      range delay for the TA 
         RDELTA=FLOAT(BUF(441)) + FLOAT(BUF(446))/10000.0 + RDELCORTA
C      gate length for the LF radar in Km 
         GTLNLF=FLOAT(BUF(143)*BUF(145))/1000.0 
         DO I=1, BUF(149) !number of LF output bins
            radiilf(i)=(gtlnlf * i) + rdellf
         END DO 
C      flag for LF variable range gates 1=VAR,0=FIX 
         IVRANGELF=BUF(142) 
         IF (IVRANGELF .EQ. 1)  
     +      WRITE(6,'(" Flag for LF variable bin spacing is on,",
     +                "**VAR. RANGES ARE UNDEFINED AT THE MOMENT**")')
C      flag for TA variable range gates 1=VAR,0=FIX 
         IVRANGETA=BUF(442) 
C      step between input bins for the Tail radar in meters 
         NRNS=BUF(443)
         IF (IVRANGETA .EQ. 1) THEN 
            WRITE(6,'(" Flag for TA variable bin spacing is on.")')
C           gate length in km 
            GTLNTA(1)=FLOAT(NRNS)/1000.0
            GTLNTA(2)= 0.150
            GTLNTA(3)= 0.300
            DO I=1, 256
               radiita(I)=gtlnta(1)*I + rdelta
            END DO
            DO I=1, 128
               radiita(I+256)=gtlnta(2)*I + radiita(256)
            END DO
            DO I=1, 128
               radiita(I+384)=gtlnta(3)*I + radiita(384)
            END DO
         ELSE 
C           buf(445) range averaging(1,2,3 or 4;undefined in var mode)
            NRNS=NRNS*BUF(445)
            GTLNTA(1)=FLOAT(NRNS)/1000.0
            GTLNTA(2)= 0.0
            GTLNTA(3)= 0.0
            DO I=1, BUF(449)
               radiita(I) = gtlnta(1)*I + rdelta
            END DO
         ENDIF
C      number of LF output bins 
         NBINLF=BUF(149)
C      number of TA output bins 
         NBINTA=BUF(449)
C      maximum range of the LF radar
         RMAXLF=FLOAT(NBINLF)*GTLNLF + RDELLF 
C      maximum range of the TA radar
         IF (IVRANGETA.EQ.0)RMAXTA=FLOAT(NBINTA)*GTLNTA(1) + RDELTA 
         IF (IVRANGETA.EQ.1)
     +    RMAXTA=(256.*GTLNTA(1))+(128.*GTLNTA(2))+(128.*GTLNTA(3))+
     +           RDELTA
         WRITE(6,'(" maximum range of the LF radar= ",F6.2,1X,/,
     +         " GTLN= ",F7.3," RDEL= ",F6.2," # bins= ",I3)')
     +   RMAXLF,GTLNLF,RDELLF,NBINLF
         WRITE(6,'(" maximum range of the TA radar= ",F6.2,1X,/,
     +         " GTLN1= ",F7.3," GTLN2= ",F7.3," GTLN3= ",F7.3, 
     +         " RDEL= ",F6.2," # bins= ",I3)') 
     +   RMAXTA,GTLNTA(1),GTLNTA(2),GTLNTA(3),RDELTA,NBINTA 
         GO TO 100
      ELSE IF (BUF(1).EQ.1 .AND. BUF(2).EQ.IERR) THEN !data record
         NIX=0
         KOUNTREC=KOUNTREC + 1
         GO TO 200
      ELSE
         WRITE(6,'(" Ierr= ",I5)') IERR 
         NIX=0
         GO TO 100
      ENDIF 
C process the data record header and the ray header 
C unpack data record
200   IDRADAR=IBITS(BUF(5),8,8) !word 5: radar# 1=LF, 2=TA
      NUMSWEEP(IDRADAR)=BUF(3)  !sweep number.
c      write(6,*)'idradar,numsweep,buf = ',
c     +  idradar,numsweep(idradar),buf(3)
C save the record numbers and corresponding times of TA and LF
C sweeps after a Tape Header.  This information is to be used by TAPESEARCH.
      CALL SAVESWEEP_DISC
      IF (IDRADAR.EQ.1) NLFSWEEP=BUF(3)
      IF (IDRADAR.EQ.2) NTASWEEP=BUF(3)
      IF (KINDRADAR.NE.IDRADAR .AND. KINDRADAR.GT.0) GO TO 100 
      NBYTESDATA=BUF(2)    !data record size in bytes.
      NWORDS= (BUF(2)+1)/2 !word2:size of record in bytes.
      NUMRECORD=BUF(4)     !record # in the swep for taht radar.
      IF (IDRADAR .EQ. 1) THEN
         NBINS=NBINLF      !number of output bins.
      ELSE
         NBINS=NBINTA 
      ENDIF 
      IRECFLAG=IBITS(BUF(5),0,8) !word 5: record # flag(1first,2middle,3last) 
      KWORDS=0                   !pointer in the data record.
      KRAYS=5                    !pointer in the ray header.
      KWORDS=KWORDS + KRAYS 
300   NIX= 0
      IERR=0
      IF (KINDRADAR.NE.IDRADAR .AND. KINDRADAR.GT.0) GO TO 100
C unpack ray header record
      NBYTESRAY=BUF(KWORDS+1)            ! ray size in bytes
      KRAYS=BUF(KWORDS+1)/2              ! 1:# of words in the ray. 
      IENDRAY=KWORDS + KRAYS
      IF (KRAYS.LE.0 .OR. KRAYS.GT.NWORDS) THEN 
          WRITE(6,'(" radar(1=LF,2=TA):",I1," corrupted size= ",I6,
     +    " rec#= ",I6," sweep#= ",I6)') IDRADAR,KRAYS,NUMRECORD,BUF(3) 
          GO TO 100 
      ENDIF 
      IF (BUF(IENDRAY) .NE. 1) THEN 
          WRITE(6,'(" radar(1=LF,2=TA):",I1," corrupted size= ",I6,
     +    "end of ray not 1"," sweep# ",I6)') 
     +     IDRADAR,KRAYS,BUF(3) 
          GO TO 100 
      ENDIF 
      ICODE=BUF(KWORDS+2)                ! 2:code/dsp/year bits13-15/11-12/0-10 
      IDSP=IBITS(BUF(KWORDS+2),11,2)     ! 2:code for which dsp data is from
      MDATA=IBITS(BUF(KWORDS+2),10,1)    ! 2:data mode code (1=time series) 
      IYEAR=IBITS(BUF(KWORDS+2),0,10)    ! 2:year(bits 0-10)
      MONTH=IBITS(BUF(KWORDS+3),8,8)     ! 3:month(bits 8-15) 
      IDAY=IBITS(BUF(KWORDS+3),0,8)      ! 3:day(bits 0-7)
      IRAYCODE=IBITS(BUF(KWORDS+4),8,8)  ! 4:ray code (bits 8-15) 
      IH=IBITS(BUF(KWORDS+4),0,8)        ! 4: hours (bits 0-7) 
      IM=BUF(KWORDS+5)                   ! 5: minutes 
      IS=FLOAT(BUF(KWORDS+6))*.01        ! 6: seconds
      IS100=FLOAT(BUF(KWORDS+6))-IS*100.0! 6: seconds(in 1/100) 
      TIME=timsec_disc(IH,IM,IS,IS100)   !time in seconds  
c      write(6,*)'ih,im,is,time = ',ih,im,is,time
      CALL CHKTM1(TIME)                   !add 24 hours over midnight.
c      write(6,*)'time = ',time
c subtract 24 hours if going back on time on tapes over midnight.
c 122400.0 is equal to 34 hours.
      IF (TIME .GT. 122400.) TIME=TIME - 86400.0
      IF (TIME .GE. 86400.0) IH=IH + 24
      ITIME(IDRADAR)=TIME
C Time series data flag 
      IF (MDATA .EQ. 1) THEN
         WRITE(6,'(" Time series data flag; not set up", 
     +             " to read time series.")')
         KWORDS=KWORDS + KRAYS
         GO TO 50 
      ENDIF   
C return if searching for a time or sweep number
      IF (SEARCH) THEN
         SEARCH=.FALSE.
         RETURN
      ENDIF
      IOLDSWEEP(IDRADAR)=NUMSWEEP(IDRADAR)
C **************
C BINARY converts 16 bit binary angles into degrees (0.005493164) 
      RLAT=FLOAT(BUF(KWORDS+7))*BINARY  ! 7:latitude(binary angle) 
      RLON=FLOAT(BUF(KWORDS+8))*BINARY  ! 8:longitude(binary angle)
      RA=BUF(KWORDS+9)                  ! 9:radar altitude(meters) 
      GSE=FLOAT(BUF(KWORDS+10))*.1      !10:E/W ground speed(1/10 m/s)
      GSN=FLOAT(BUF(KWORDS+11))*.1      !11:N/S ground speed(1/10 m/s)
      GS=SQRT(GSN*GSN + GSE*GSE)        !ground speed (m/s) 
      VGS=FLOAT(BUF(KWORDS+12))*.1      !12:vert. aircraft speed(1/10 m/s)
      UCOM=FLOAT(BUF(KWORDS+13))*.1     !13:U comp(1/10 m/s)
      VCOM=FLOAT(BUF(KWORDS+14))*.1     !14:V comp(1/10 m/s)
      WD=WDIR1(UCOM,VCOM)                !wind direction(deg)
      WS=SPEED1(UCOM,VCOM)               !wind speed(m/s)
      VWS=FLOAT(BUF(KWORDS+15))*.1      !15:vertical wind speed(1/10 m/s) 
      IRCU=FLOAT(BUF(KWORDS+16))        !16:RCU status
      ELEV=FLOAT(BUF(KWORDS+17))*BINARY !17:elevation(binary angle) 
      AZM=FLOAT(BUF(KWORDS+18))*BINARY  !18:azimuth(binary angle) 
      PITCH=FLOAT(BUF(KWORDS+19))*BINARY!19:pitch(binary angle,+ front up 
      ROLL=FLOAT(BUF(KWORDS+20))*BINARY !20:roll(binary angle,+ right up) 
      DRIFT=FLOAT(BUF(KWORDS+21))*BINARY!21:drift angle 
      HEADING=FLOAT(BUF(KWORDS+22))*BINARY !22:heading(binary angle) 
      IF (AZM .LT. 0.0) AZM=AZM + 360.0 
      IF (HEADING .LT. 0.0) HEADING= HEADING + 360.0 
C return if the user does not want to read the data
      IF (IDATAFLAG .EQ. 0) THEN
         KWORDSAVE=KWORDS
         KWORDS=KWORDS + KRAYS
         RETURN
      ENDIF
C unpack and uncompress the data
400   CALL DCOMPRESS1(ICODE,BUF(KWORDS+23),IZ,IV,IW) 
C the uncorrected reflectivity is computed as dBZ=(n-64)/2. 
C   0: indicates no reflectivity
C   1: -31.5 dBZ
C  64:   0.0 dBZ
C 255: +95.5 dBZ
C velocities is computed as v=(n-128)*(xniqta/128.0)
C   0: indicates vel. data not available at this range
C   1: maximum vel. towards the radar 
C 128: zero velocity
C 255: maximum velocity away from the radar 
C spectral width is computed as w=(n/256)*xniqta (m/s). 
C   0: indicates that width data was not avilable 
C   1: minimum fraction of variance 
C 255: maximum fraction of variance 
C The Spectral Width is converted to meters per second when multipy 
C by the unambigous velocity.
C Data
      DO I=1,NBINS
         IF (ibits(idataflag,2,1) .eq. 1) then
            IF (IZ(I) .EQ. 0) THEN 
               DBZBUFFER(I)= -32.0
            ELSE 
               DBZBUFFER(I)=FLOAT(IZ(I)-64)*0.5
            ENDIF
         ENDIF
         IF (ibits(idataflag,1,1) .eq. 1) then
            IF (IV(I) .EQ. 0) THEN 
               VELBUFFER(I)=-888.88
            ELSE 
               VELBUFFER(I)=FLOAT(IV(I)-128)*VELCONV_TA
            ENDIF
         ENDIF
         IF (ibits(idataflag,0,1) .eq. 1) then
            IF (IW(I) .EQ. 0) THEN
               WIDTHBUFFER(I)=0.0
            ELSE
               WIDTHBUFFER(I)=(FLOAT(IW(I))*.00390625)*XNIQTA
            ENDIF
         ENDIF
      END DO
      KWORDS=KWORDS + KRAYS 
      RETURN
C
C entry to get data
      ENTRY RADARDATA_DISC(IDATAFLAG)
      KWORDS=KWORDSAVE
      GO TO 400
C *****
      entry openradarfile(nameradarfile,lut,lup,ierr)
C
C This subroutine opens a disc file containing radar data.  The radar data
C is stored in the Sigmet format.  The files were created by 2fb or 
C cpreaddisc.
C nameradarfile - radar disc filename; format:sub-directory/filename
C                 there is no need to specify /hrd/dat if the file 
C                 resides there.
C lut - disc file logical number
C lup - output display[default to user's screen]
C ierr - error condition flag: 0: no errors
C                             >0: error condition exists
C        The I/O status error are listed and described in the HP 700/800
C        FORTRAN90 Programmer Reference in Appendix C.
C
C logical unit lut is connected to the file
       lp=lup
       open(lut,FILE=nameradarfile,
     +      err=1200,IOSTAT=ios,FORM='unformatted',STATUS='old')
C error messages:
       return
1200   write(lp,'("Error opening file: ",i6)') ios
       end
C *****
      function timsec_disc(ih,im,is,is100)
C Function that converts hours, minutes, seconds and hundred of seconds
C into seconds.
C ih - hours
C im - minutes
C is - seconds
C is100 - hundred of seconds
      timsec_disc=float(ih)*3600.0+float(im)*60.0+
     +                       (float(is)+float(is100)/100.)
      return
      end
c *****
      subroutine savesweep_disc
c   This subroutine saves the record number of the start of every sweep
c and its corresponding time after a Tape Header.
      COMMON /TSEARCH/ kountrec,firstsweep(2),keylf(1024,2),
     +                 keyta(4096,2),
     +                 numsweep(2),itime(2),lastpoint(2),
     +                 iheadertime,ioldsweep(2),kwords,nwords,krays,buf
      integer*2 buf(4096)
      integer firstsweep
      integer keepsweep(2)
      data keepsweep/2*0/
c
      idradar=ibits(buf(5),8,8)          !word 5: radar# 1=LF, 2=TA
      if (kountrec.eq.1) then
         firstsweep(idradar)=buf(3)      !1st sweep after header
         keepsweep(1)= 0                 !previous sweep number; set to 0
         keepsweep(2)= 0                 !after a new header is read.
      endif
c update information after a new sweep number is read
      if (keepsweep(idradar).ne.numsweep(idradar)) then
         if(idradar.eq.1.and.firstsweep(1).eq.0)
     +     firstsweep(1)=numsweep(1)  !1st LF sweep after header.
         if(idradar.eq.2.and.firstsweep(2).eq.0)
     +     firstsweep(2)=numsweep(2)  !1st TA sweep after header.
         ih=ibits(buf(9),0,8)               ! 4: hours (bits 0-7) 
         im=buf(10)                         ! 5: minutes 
         is=float(buf(11))*.01              ! 6: seconds
         is100=float(buf(11))-is*100.0      ! 6: seconds(in 1/100) 
         time=timsec_disc(ih,im,is,is100)   !time in seconds  
c         call chktm(time)                   !add 24 hours over midnight.
c subtract 24 hours if going back on time on tapes over midnight.
c 122400.0 is equal to 34 hours.
         if (time .gt. 122400.) time=time-86400.0
         if (time .ge. 86400.0) ih=ih + 24
         itime(idradar)=time                !save time for searching.
C **************
C Store record numbers of start of sweeps and its corresponding times
c        write(6,'(/3i2.2,"saving radar (1=lf,2=ta):",i1," sweep#",i6)')
c    +             ih,im,is,idradar,numsweep(idradar)
         if (idradar.eq.1) then
           ipoint=numsweep(1)-firstsweep(1)+1        !pointer in lf.
           write(6,*)'ipoint,numsweep,firstsweep(1) = ',
     +                ipoint,numsweep(1),firstsweep(1)
           if (ipoint.gt.1024) then
c             1024 is the max. number of lf sweeps within two tape headers.
              write(6,'(" **increase keylf size**")') 
              stop
           endif
           if(ipoint.ge.1)then
            keylf(ipoint,1)=kountrec                  !store record # of sweep.
            keylf(ipoint,2)=itime(1)                  !store time of sweep.
            lastpoint(1)=ipoint                       !last pointer on keylf.
           endif
         else
           ipoint=numsweep(2)-firstsweep(2)+1        !pointer in lf. 
           if (ipoint.gt.4096) then
c             4096 is the max. number of ta sweeps within two tape headers.
              write(6,'(" **increase keyta size**")')
              stop
           endif
           if(ipoint.ge.1)then
            keyta(ipoint,1)=kountrec                  !store record # of sweep.
            keyta(ipoint,2)=itime(2)                  !store time of sweep.
            lastpoint(2)=ipoint                       !last pointer on keyta.
           endif
         endif
         keepsweep(idradar)=numsweep(idradar)        !update previous sweep.
      endif
      return
      end
c *****
      subroutine discsearch(lut,kindradar,iop,rkey,ierr)
c
c   This subroutine reads the new format data and searches for a 
c specified sweep number or a specified time.  It positions the disc at 
c the begining of the specified sweep number or the specified time.  
c Tapesearch should be used in conjunction with Radardiscread.  It will
c not work by itself.  Tapesearch assumes that a Tape Header has been read.
c Not reading a Tape Header will produce unpredictable results!!!!
c The calling routine sends:
c lut - disc logical unit number               (integer)
c kindradar - kind of radar 1:lf, 2:ta         (integer)
c iop - (1) search by sweep number             (integer)
c       (2) search by time
c rkey - sweep number to search for    (iop=1) (real)
c        time to search for in seconds (iop=2)
c tapesearch returns:
c ierr - error code or size of record in bytes
c         0 no error; good read
c        -2 for end of file, final record on the disc file
c
c variables on TSEARCH:
c     kountrec - record counter.
c     firstsweep - buffer containing the 1st LF and TA sweep # after a header.
c     keylf,keyta - buffers containing the record number of the start of 
c                   every sweep and its corresponding times.
c     numsweep - current sweep number for lf and ta.
c     itime - array holding the time in seconds for lf and ta.
c     lastpoint - last pointer on keylf and keyta
c     iheadertime - header time in seconds.
c     ioldsweep - previous sweep number for lf and ta.
      COMMON /TSEARCH/ kountrec,firstsweep(2),keylf(1024,2),
     +                 keyta(4096,2),
     +                 numsweep(2),itime(2),lastpoint(2),
     +                 iheadertime,ioldsweep(2),kwords,nwords,krays,buf
      COMMON /IERRINONE/ ierrin
      integer*2 buf(4096)
      integer firstsweep
      logical search, search1
c
      nbackfiles=2                    !initialize # of files to go back.
      iheadertimeo=-1                 !initialize previous header time.
      search=.false.                  !not searching.
      key=rkey                        !convert sweep # or time to integer.
      ierrin=ierr
c      write(6,*)'ierrin = at discsearch = ',ierrin
      ierr=0                          !reset error code
      if (iop.ne.1 .and. iop.ne.2) then
         write(6,'(" illegal search option= ",i6)') iop
         return
      endif
      if (kindradar.lt.1 .or. kindradar.gt.2) then
         write(6,'(" wrong radar id specified")') kindradar
         return
      endif
      go to (20,40), iop
c search for a sweep number
c     sweep number beyond current position
20    if (key .gt. numsweep(kindradar)) then 
         nwords= -1
         go to 100
c     sweep number within current block
      else if (key.ge.firstsweep(kindradar) .and. 
     +         key.le.numsweep(kindradar)) then      
         nwords= -1
         rewind lut
         go to 100
c     sweep before current block
      else if (key .lt. firstsweep(kindradar)) then
         nwords= -1
         rewind lut
         write(6,'(" searching for=",i6," sweep# ",i6,
     +             " 1st sweep after header=",i6)')
     +   key,numsweep(kindradar),firstsweep(kindradar)
         go to 100
      endif
c search by time
c     time beyond current position
c40    write(6,*)'key,itime,kindradar = ',key,itime(kindradar),kindradar
40    if (key .gt. itime(kindradar)) then
         go to 100
      else
         rewind lut
         nwords= -1
         call ctme1(rkey,iht,imt,ist)
         call ctme1(float(itime(kindradar)),iht1,imt1,ist1)
         write(6,'(" searching for=",3i2.2," read time ",3i2.2)')
     +      iht,imt,ist,iht1,imt1,ist1
         go to 100
      endif
c
c
c read the disc file
100   call discentry(lut,kindradar,0,.true.,ierr)
      if (ierr .lt. 0) then
         write(6,'(" end of data: ",i6)') ierr
         return
      endif
c display message whenever sweep changes.
300   if (ioldsweep(kindradar) .ne. numsweep(kindradar)) then
         idradar=ibits(buf(5),8,8) !word 5: radar# 1=LF, 2=TA
         call ctme1(float(itime(kindradar)),iht,imt,ist)
         write(6,'(1x,3i2.2," radar (1=lf,2=ta):",i1," sweep#",i6)')
     +            iht,imt,ist,idradar,numsweep(kindradar)
         ioldsweep(kindradar)=numsweep(kindradar)
      endif
c search for a time
      if (iop .eq. 2) then
         if (itime(kindradar).lt.key) then
            kwords=kwords + krays
            go to 100
         else
            return
         endif
c search for a specified sweep number
      else if (iop .eq. 1) then
         if (numsweep(kindradar).ge.key)then
            kwords=5
            return
         endif
         nwords= -1
         go to 100
      endif
      return 
      end
C *****
      subroutine radardiscrewind(lut)
c This subroutine rewinds the 1988 radar disc files
      COMMON /TSEARCH/ kountrec,firstsweep(2),keylf(1024,2),
     +                 keyta(4096,2),
     +                 numsweep(2),itime(2),lastpoint(2),
     +                 iheadertime,ioldsweep(2),kwords,nwords,krays,buf
      integer*2 buf(4096)
      integer firstsweep
      write(6,'(" rewinding file!!")')
      rewind lut
      kwords=0
      nwords=-1
      return
      end
c *****
      subroutine getsweepn_disc(lut,kindradar,isweepn,ierr)
c
c   This subroutine reads the new format data and positions the file at 
c the Nth sweep.
c Getsweepn should be used in conjunction with Radardiscread.  It will
c not work by itself.  Getsweepn assumes that a Tape Header has been read.
c Not reading a Tape Header will produce unpredictable results!!!!
c The calling routine sends:
c lut - disc logical unit number               (integer)
c kindradar - kind of radar 1:lf, 2:ta         (integer)
c isweepn - all,second,third,fourth..Nth sweep (integer)
c getsweepn returns:
c ierr - error code or size of record in bytes
c         0 no error; good read
c        -2 for end of file, final record on the file
c
c variables on TSEARCH:
c     kountrec - record counter.
c     firstsweep - buffer containing the 1st LF and TA sweep # after a header.
c     keylf,keyta - buffers containing the record number of the start of 
c                   every sweep and its corresponding times.
c     numsweep - current sweep number for lf and ta.
c     itime - array holding the time in seconds for lf and ta.
c     lastpoint - last pointer on keylf and keyta
c     iheadertime - header time in seconds.
c     ioldsweep - previous sweep number for lf and ta.
      COMMON /RAYHEADER / nbytesdata,nlfsweep,ntasweep,numrecord,
     +                    irecflag,idradar,nbytesray,icode,idsp,mdata,
     +                    iyear,month,iday,iraycode,time,ih,im,
     +                    is,is100,rlat,rlon,ra,gse,gsn,gs,vgs,
     +                    ucom,vcom,wd,ws,vws,ircu,
     +                    elev,azm,pitch,roll,drift,heading,
     +                    dbzbuffer(512),velbuffer(512),widthbuffer(512)
      COMMON /TSEARCH/ kountrec,firstsweep(2),keylf(1024,2),
     +                 keyta(4096,2),
     +                 numsweep(2),itime(2),lastpoint(2),
     +                 iheadertime,ioldsweep(2),kwords,nwords,krays,buf
      integer*2 buf(4096)
      integer firstsweep
c
      if (isweepn .le. 0) return
c read the file
100   call discentry(lut,kindradar,0,.true.,ierr)
      if (ierr .lt. 0) then
         write(6,'(" end of data: ",i6)') ierr
         return
      endif
      if (mod(ntasweep,isweepn) .ne. 0) then
         nwords= -1
         go to 100
      else
         kwords=5
         return
      endif
      return
      end
c      FUNCTION SPEED(U,V)
c      IF(U.EQ.0..AND.V.EQ.0.)THEN
c       SPEED=0.
c      ELSE
c       SPEED=SQRT(U*U+V*V)
c      ENDIF
c      RETURN
c      END
c      FUNCTION WDIR(U,V)
c      IF(U.EQ.0..AND.V.EQ.0.)THEN
c       WDIR=0.
c      ELSE
c       WDIR=180./3.14159*ATAN2(-U,-V)
c       IF(WDIR.LT.0.)WDIR=WDIR+360.
c      ENDIF
c      RETURN
c      END
      FUNCTION SPEED1(U,V)
      IF(U.EQ.0..AND.V.EQ.0.)THEN
       SPEED1=0.
      ELSE
       SPEED1=SQRT(U*U+V*V)
      ENDIF
      RETURN
      END
      FUNCTION WDIR1(U,V)
      IF(U.EQ.0..AND.V.EQ.0.)THEN
       WDIR1=0.
      ELSE
       WDIR1=180./3.14159*ATAN2(-U,-V)
       IF(WDIR1.LT.0.)WDIR1=WDIR1+360.
      ENDIF
      RETURN
      END
      SUBROUTINE CHKTM1(TIMES)
      DATA OLDTIMES/-1./,ICHNGTM/0/,ICHNGCOUNTER/0/
      SAVE OLDTIMES,ICHNGTM,ICHNGCOUNTER
      TIMEDIFF=TIMES-OLDTIMES
      IF(TIMEDIFF.LT.0.)THEN
C       WRITE(6,*)'TIMES,OLDTIMES,ICHNGTM,ICHNGCOUNTER = ',
C     1  TIMES,OLDTIMES,ICHNGTM,ICHNGCOUNTER
c       PAUSE
      ENDIF
      IF(TIMEDIFF.LT.-86400..AND.ICHNGCOUNTER.LT.50)THEN
       ICHNGTM=0
       ICHNGCOUNTER=0
       WRITE(6,*)'RESETTING ICHNGTM,ICHNGCOUNTER'
      ELSEIF(TIMEDIFF.LT.-80000.)THEN
       ICHNGTM=1
c       WRITE(6,*)'TIMES,OLDTIMES = ',TIMES,OLDTIMES
c       PAUSE
      ENDIF 
      IF(ICHNGTM.EQ.1)THEN
       TIMES=TIMES+86400.
       ICHNGCOUNTER=ICHNGCOUNTER+1
      ENDIF
      OLDTIMES=TIMES
      RETURN
      END

