function [ newcolmat ] = changecolormap( colorvector, colorscheme, varargin )

% Usage: [ newcolmat ] = changecolormap( colorvector, colorscheme, {whiterange,newcolorscheme,doreverse/schemetype,colorrepeat,colormanual} )
%
% Written by Altug Aksoy, April 2009
%
% This function determines a new color map based on input vector and
% colorscheme specification.
%
% Available color schemes:
% For colorscheme=1:  Change "jet" color scheme so that a plot with non-symmetric limits can be plotted with same zero color and color spread
%                     (For regular "jet" zero color is in the middle)
% For colorscheme=2:  Change "gray" color scheme so that zero is white
% For colorscheme=3:  Change "gray" color scheme so that zero is black
% For colorscheme=4:  Change to standard dbz reflectivity color scheme
% For colorscheme=5:  Change to standard radial velocity color scheme
% For colorscheme=6:  Same as colorscheme=1 with "jet" color, but around zero is NOT converted to white
% For colorscheme=7:  All warm colors with white around zero
% For colorscheme=8:  All warm colors with no white around zero
% For colorscheme=9:  All cold colors with white around zero
% For colorscheme=10: All cold colors with no white around zero
% For colorscheme=11: Predefined Matlab color scheme (string; see help with 'colormap' function; see optional arguments 2 and 3 that also apply)
% For colorscheme=12: Predefined Matlab color scheme with white around zero
%                     (string; see help with 'colormap' function; see optional arguments 2 and 3 that also apply)
% For colorscheme=13: [r1,g1,b1],[r2,g2,b2] User-defined end colors to shade in-between (vector; see optional arguments 2 and 3 that also apply)
% For colorscheme=14: [r1,g1,b1],[r2,g2,b2] User-defined end colors to shade in-between with white around zero
%                     (vector; see optional arguments 2 and 3 that also apply)
%
% whiterange:         (Optional 1) For schemes that have white for zero, +/- range that is white
%                       -Default is one countour level on either side
%                       -Otherwise specify scalar that represents the white range in number of contour levels (+/- contours will be white)
%                       -Should be greater than one
% newcolorscheme:     (Optional 2) For color schemes 11-12, name of the Matlab colormap scheme (also specify doreverse)
%                                  For color schemes 13-14, the two user-specified end colors {[r/h1,g/s1,b/v1],[r/h2,g/s2,b/v2]}
%                                                           (alsp specify schemetype for RGB vs HSV color schemes)
%                       -This is the second optional argument, you must specify "whiterange" also
% doreverse:          (Optional 3) For options 11-12: If newcolorscheme is specified, whether to reverse it (0: no-default, 1:yes-reverse)
%                       -This is the third optional argument, you must specify "whiterange" and "newcolorscheme" also
% schemetype:         (Optional 3) For options 13-14: If newcolorscheme is specified, what color scheme to use (0: RGB-default, 1:HSV)
%                       -This is the third optional argument, you must specify "whiterange" and "newcolorscheme" also
% colorrepeat:        (Optional 4) For options 13-14: Whether to use user-specified increments to repeat colors (0=no special increment, X=user increment)
%                       -This is the fourth optional argument, you must specify "whiterange", "newcolorscheme", and "schemetype" also
% colormanual:        (Optional 5) For any colorscheme option: If specified as the fifth option after the regular ones above, manually change colors
%                                                              by specifying [[colorindstart1,colorindend1,schemetype1(0/1),r1/h1,g1/s1,b1/v1];[colorindstart2,colorindend2,schemetype2(0/1),r2/h2,g2/s2,b2/v2];...]
%                                                              (Specify in-between options as empty vectors [] if necessary)
%                                                                 --colorindstart: starting color index to change
%                                                                 --colorindend  : ending color index to change
%                                                                 --schemetype   : 0:rgb, 1:hsv
%                                                                 --color        : [r,g,b] or [h,s,v] triplet
%                                                                 ----> specify as many groups manually as you like
%                                                                 ----> should be a vector of n x 6 where n is the number of special colors specified

if (nargin < 2)
    error('At least two arguments are expected.')
    return
end

colormanual = [];
if (nargin == 2)
    whiterange = 1;
elseif (nargin > 2)
    whiterange = varargin{1};
    whiterange = max([1 whiterange]);
    
    newclrscheme = [];
    doreverse    = [];
    if (nargin > 3)
        newclrscheme = varargin{2};
    end
    if (nargin > 4)
        doreverse    = varargin{3};
        schemetype   = varargin{3};
        colorrepeat  = varargin{4};
    end
    
    if (nargin == 8)
        colormanual  = varargin{6};
    end
end

if any(colorscheme == [11 12])
    if (isempty(newclrscheme) | isempty(doreverse))
        error('When a new color scheme is defined through options 11 or 12, neither newcolorscheme -input 4- nor doreverse -input 5- can be empty.')
    elseif ~isstring(newclrscheme)
        error('When a new color scheme is defined through options 11 or 12, newcolorscheme -input 4- must be a string.')
    end
end

if any(colorscheme == [13 14])
    if (isempty(newclrscheme))
        error('When a new color scheme is defined through options 13 or 14, newcolorscheme -input 4- cannot be empty.')
    elseif ~(length(newclrscheme) == 6)
        error('When a new color scheme is defined through options 13 or 14, newcolorscheme -input 4- must be a vector of length six.')
    elseif (isempty(colorrepeat))
        colorrepeat = 0;
    end
end

docolormanual = 0;
if ~isempty(colormanual)
    docolormanual  = 1;
    numcolormanual = size(colormanual,1);
    numargument    = size(colormanual,2);
    if (numargument ~= 6 )
        error('For the fifth argument to be valid, each row needs to be of length six.');
    end
end

% "jet" opens a figure if there's no figure open already
% in order to prevent an already open figure, need to figure out
% if there's already an existing figure
fnow = isempty(get(groot,'CurrentFigure'));

if (colorscheme <= 10)
    numallcolors = size(jet,1);if fnow;close gcf;end
elseif any(colorscheme == [11 12])
    schemenow    = feval(newclrscheme);
    if doreverse
        schemenow = flipud(schemenow);
    end
    numallcolors = size(schemenow,1);if fnow;close gcf;end
else
    % This is dummy for these options
    numallcolors = size(jet,1);if fnow;close gcf;end
end

numhalf      = numallcolors/2;
numhalfmin1  = numhalf;
numhalfpls1  = numhalf;

caxismin     = min(colorvector);
caxismax     = max(colorvector);
colorsize    = length(colorvector) - 1;

if any(colorscheme == [1 2 3 6 7 8 9 10 11 12])
    %hh = figure('Visible','off');
    if any(colorscheme == [1 6 7 8 9 10])
        colmap       = jet;if fnow;close gcf;end
        do_zeroshift = any(colorscheme == [1 7 9]);
    elseif any(colorscheme == [11 12])
        colmap       = schemenow;if fnow;close gcf;end
        do_zeroshift = (colorscheme == 12);
    elseif (colorscheme == 2)
        colmap = zeros(numallcolors,3);
        for i = 1:numhalf
            colornow = 0.2 + 0.8/numhalfmin1*(i-1);
            colmap(i,:) = [colornow colornow colornow];
        end
        for i = 1:numhalf
            colornow = 1 - 0.8/numhalfmin1*(i-1);
            colmap(numhalf+i,:) = [colornow colornow colornow];
        end
        do_zeroshift = 1;
    elseif (colorscheme == 3)
        colmap = zeros(numallcolors,3);
        for i = 1:numhalf
            colornow = 1 - 0.8/numhalfmin1*(i-1);
            colmap(i,:) = [colornow colornow colornow];
        end
        for i = 1:numhalf
            colornow = 0.2 + 0.8/numhalfmin1*(i-1);
            colmap(numhalf+i,:) = [colornow colornow colornow];
        end
        do_zeroshift = 0;
    end
    %close(hh);
    
    % Set white zero range

    % Check to see if "0" contour is within the color vector - set location of
    % "zero" and contour locations around "zero"
    if find(colorvector == 0)
        zeroposition = find(colorvector == 0);
        zeroshift    = [-whiterange +whiterange];
    elseif (caxismax < 0)
        zeroposition = colorsize-1;
        zeroshift    = [-whiterange 0];
    elseif (caxismin > 0)
        zeroposition = 1;
        zeroshift    = [0 +whiterange];
    else
        zeroposition = min(find(colorvector > 0)) - 1;
        zeroshift    = [-whiterange +whiterange];
    end

    zeroindices = max(1,zeroposition+zeroshift(1)) : min(zeroposition+zeroshift(2)-1,colorsize);

    newcolmat = zeros(length(colorvector)-1,3);
    % Determine appropriate limits for regular "jet"
    if any(colorscheme == [7 8])
        colorpositive  = colorsize - 1;
        newcolorscale2 = numhalfpls1:numhalfmin1/max([(colorpositive-1) 1]):numallcolors;
        doall          = 0;
        dofirst        = 0;
        dosecond       = 1;
        index2         = numhalfpls1:numallcolors;
        index2n        = 1:max([colorpositive 2]);
    elseif any(colorscheme == [9 10])
        colornegative  = colorsize - 1;
        newcolorscale1 = 1:numhalfmin1/max([(colornegative-1) 1]):numhalf;
        doall          = 0;
        dofirst        = 1;
        dosecond       = 0;
        index1         = 1:numhalf;
        index1n        = 1:max([colornegative 2]);
    elseif any(colorscheme == [11 12])
        doall          = 1;
        dofirst        = 0;
        dosecond       = 0;
        indexall       = 1:numallcolors;
        newcolorscale  = 1:numallcolors/colorsize:numallcolors;
    else
        if (caxismin<0 & caxismax>0) & (zeroposition > 2)
            %colorpositive  = max([(colorsize - 1 - zeroposition) 2]);
            colorpositive  = max([(colorsize - zeroposition) 2]);
            colornegative  = max([(zeroposition) 2]);
            newcolorscale1 = 1:numhalfmin1/max([(colornegative) 1]):numhalf;
            newcolorscale2 = numhalf:numhalfmin1/max([(colorpositive-1) 1]):numallcolors;
            doall          = 0;
            dofirst        = 1;
            dosecond       = 1;
            index1         = 1:numhalf;
            index2         = numhalfpls1:numallcolors;
            index1n        = 1:colornegative;
            index2n        = colornegative+1:colornegative+colorpositive;
        elseif (caxismax<=0)
            %colornegative  = colorsize - 1;
            colornegative  = colorsize;
            newcolorscale1 = 1:numhalfmin1/max([(colornegative-1) 1]):numhalf;
            doall          = 0;
            dofirst        = 1;
            dosecond       = 0;
            index1         = 1:numhalf;
            index1n        = 1:max([colornegative 2]);
        else
            %colorpositive  = colorsize - 1;
            colorpositive  = colorsize;
            newcolorscale2 = numhalfpls1:numhalfmin1/max([(colorpositive-1) 1]):numallcolors;
            doall          = 0;
            dofirst        = 0;
            dosecond       = 1;
            index2         = numhalfpls1:numallcolors;
            index2n        = 1:max([colorpositive 2]);
        end
    end

    if (doall == 1);
        rednew   = interp1(indexall,squeeze(colmap(indexall,1)),newcolorscale);
        greennew = interp1(indexall,squeeze(colmap(indexall,2)),newcolorscale);
        bluenew  = interp1(indexall,squeeze(colmap(indexall,3)),newcolorscale);
        newcolmat(1:colorsize,1) = rednew;
        newcolmat(1:colorsize,2) = greennew;
        newcolmat(1:colorsize,3) = bluenew;
    end

    if (dofirst == 1);
        rednew   = interp1(index1,squeeze(colmap(index1,1)),newcolorscale1);
        greennew = interp1(index1,squeeze(colmap(index1,2)),newcolorscale1);
        bluenew  = interp1(index1,squeeze(colmap(index1,3)),newcolorscale1);
        newcolmat(index1n,1) = rednew;
        newcolmat(index1n,2) = greennew;
        newcolmat(index1n,3) = bluenew;
    end

    if (dosecond == 1);
        rednew   = interp1(index2,squeeze(colmap(index2,1)),newcolorscale2);
        greennew = interp1(index2,squeeze(colmap(index2,2)),newcolorscale2);
        bluenew  = interp1(index2,squeeze(colmap(index2,3)),newcolorscale2);
        newcolmat(index2n,1) = rednew;
        newcolmat(index2n,2) = greennew;
        newcolmat(index2n,3) = bluenew;
    end

    if do_zeroshift
        for izero = zeroindices
            newcolmat(izero,:) = [1 1 1];
        end
    end
    
elseif (colorscheme == 4)
    refcolors( 1,:) = [  0   0   0] / 255;
    refcolors( 2,:) = [204 255 255] / 255;
    refcolors( 3,:) = [204 153 204] / 255;
    refcolors( 4,:) = [153 102 153] / 255;
    refcolors( 5,:) = [102  51 102] / 255;
    refcolors( 6,:) = [204 204 153] / 255;
    refcolors( 7,:) = [153 153 102] / 255;
    refcolors( 8,:) = [102 102  51] / 255;
    refcolors( 9,:) = [102 255 255] / 255;
    refcolors(10,:) = [51  153 255] / 255;
    refcolors(11,:) = [  0   0 255] / 255;
    refcolors(12,:) = [  0 255   0] / 255;
    refcolors(13,:) = [  0 204   0] / 255;
    refcolors(14,:) = [  0 153   0] / 255;
    refcolors(15,:) = [255 255   0] / 255;
    refcolors(16,:) = [255 204   0] / 255;
    refcolors(17,:) = [255 104   0] / 255;
    refcolors(18,:) = [255   0   0] / 255;
    refcolors(19,:) = [204  51   0] / 255;
    refcolors(20,:) = [153   0   0] / 255;
    refcolors(21,:) = [255   0 255] / 255;
    refcolors(22,:) = [153  51 204] / 255;
    refcolors(23,:) = [255 255 255] / 255;
    refcolors(24,:) = [255 255 255] / 255;
    
    reflimits( 1)   = -30;
    reflimits( 2)   = -28;
    reflimits( 3)   = -25;
    reflimits( 4)   = -20;
    reflimits( 5)   = -15;
    reflimits( 6)   = -10;
    reflimits( 7)   =  -5;
    reflimits( 8)   =   0;
    reflimits( 9)   =   5;
    reflimits(10)   =  10;
    reflimits(11)   =  15;
    reflimits(12)   =  20;
    reflimits(13)   =  25;
    reflimits(14)   =  30;
    reflimits(15)   =  35;
    reflimits(16)   =  40;
    reflimits(17)   =  45;
    reflimits(18)   =  50;
    reflimits(19)   =  55;
    reflimits(20)   =  60;
    reflimits(21)   =  65;
    reflimits(22)   =  70;
    reflimits(23)   =  75;
    reflimits(24)   =  80;
    
    indexstart      = find(reflimits == colorvector(1));
    indexend        = find(reflimits == colorvector(end));
    
    refcolorsred   = squeeze(refcolors(indexstart:indexend,1))';
    refcolorsgreen = squeeze(refcolors(indexstart:indexend,2))';
    refcolorsblue  = squeeze(refcolors(indexstart:indexend,3))';
    
    numallcolors   = indexend - indexstart + 1;
    
    indexall       = 1:numallcolors;
    newcolorscale  = linspace(1,numallcolors,colorsize);

    rednew   = interp1(indexall,refcolorsred,  newcolorscale);
    greennew = interp1(indexall,refcolorsgreen,newcolorscale);
    bluenew  = interp1(indexall,refcolorsblue, newcolorscale);
    
    newcolmat(1:colorsize,1) = rednew;
    newcolmat(1:colorsize,2) = greennew;
    newcolmat(1:colorsize,3) = bluenew;
    
elseif (colorscheme == 5)
    numshades = (length(colorvector)-1)/2;
    redshades = linspace(0.3,1.0,(numshades-1));
    grnshades = linspace(1.0,0.3,(numshades-1));
    for i=1:(numshades-1)
        newcolmat(i,:) = [0.000 grnshades(i) 0.000];
    end
    newcolmat(numshades  ,:) = [0.700 0.850 0.700];
    newcolmat(numshades+1,:) = [0.850 0.700 0.700];
    for i=1:(numshades-1)
        newcolmat(numshades+1+i,:) = [redshades(i) 0.000 0.000];
    end
    
elseif any(colorscheme == [13 14])
    do_zeroshift = (colorscheme == 14);
    % x/y/z either r/g/b or h/s/v values
    [x1] = newclrscheme(1);
    [y1] = newclrscheme(2);
    [z1] = newclrscheme(3);
    [x2] = newclrscheme(4);
    [y2] = newclrscheme(5);
    [z2] = newclrscheme(6);
    
    xvalues = linspace(x1,x2,colorsize);
    yvalues = linspace(y1,y2,colorsize);
    zvalues = linspace(z1,z2,colorsize);
    
    if ~schemetype
        % RGB - no conversion needed
        newcolmatx(:,1) = xvalues;
        newcolmatx(:,2) = yvalues;
        newcolmatx(:,3) = zvalues;
    else
        % HSV - convert to RGB first
        rgbvalues      = hsv2rgb([xvalues',yvalues',zvalues']);
        newcolmatx(:,1) = rgbvalues(:,1);
        newcolmatx(:,2) = rgbvalues(:,2);
        newcolmatx(:,3) = rgbvalues(:,3);
    end
    
    if colorrepeat
        % Distribute colors at user-specified intervals
        indnow = 1;
        ccvals = [caxismin:colorrepeat:caxismax];
        for cvals=ccvals(1:end-1)
            [ccol,icol] = find(colorvector<=cvals);
            newcolmat(indnow,1:3) = newcolmatx(max(icol),1:3);
            indnow = indnow + 1;
        end
    else
        newcolmat = newcolmatx;
    end

    if do_zeroshift
        for izero = zeroindices
            newcolmat(izero,:) = [1 1 1];
        end
    end
    
end

% Finally change entries manually if requested
if docolormanual
    for icol = 1:numcolormanual
        currentopt = squeeze(colormanual(icol,:));
        indexstart = currentopt(1);
        indexend   = currentopt(2);
        dohsv      = currentopt(3);
        colornow   = currentopt(4:6);
        if ~dohsv
            for numindchange = indexstart:indexend
                newcolmat(numindchange,:) = colornow;
            end
        else
            rgbvalues = hsv2rgb(colornow);
            for numindchange = indexstart:indexend
                newcolmat(numindchange,:) = rgbvalues;
            end
        end
    end
end


